const sqlite3 = require('sqlite3').verbose();
const fs = require('fs');
const readline = require('readline');
const path = require('path');
const dbFilePath = path.join(__dirname, 'proxies.db'); // Mengarahkan ke folder 'proxy'


// Fungsi untuk memastikan database ada
function ensureDatabaseExists() {
    if (!fs.existsSync(dbFilePath)) {
        console.error(`Database file ${dbFilePath} does not exist.`);
        process.exit(1); // Keluar dari aplikasi jika database tidak ditemukan
    }
}

// Fungsi untuk membuat tabel jika belum ada
function ensureTablesExist(db) {
    db.serialize(() => {
        db.run(`
      CREATE TABLE IF NOT EXISTS proxies (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        proxy_list_id INTEGER,
        address TEXT,
        used INTEGER,
        status TEXT,
        FOREIGN KEY (proxy_list_id) REFERENCES proxy_list (id)
      )
    `);

        db.run(`
      CREATE TABLE IF NOT EXISTS proxy_list (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        start TEXT,
        end TEXT
      )
    `);

        db.run(`
      CREATE TABLE IF NOT EXISTS used_at (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        proxy_id INTEGER,
        times TEXT,
        status TEXT,
        FOREIGN KEY (proxy_id) REFERENCES proxies (id)
      )
    `);
    });
}

function ensureProxyListDefault(db) {
    db.serialize(() => {
        // Mengecek apakah sudah ada baris dengan id = 1 di tabel proxy_list
        db.get(`SELECT * FROM proxy_list WHERE id = 1`, (err, row) => {
            if (err) {
                return console.error(`Error querying proxy_list: ${err.message}`);
            }

            // Jika belum ada baris dengan id = 1, tambahkan data default dengan start dan end bernilai NULL
            if (!row) {
                db.run(`INSERT INTO proxy_list (start, end) VALUES (NULL, NULL)`, function (err) {
                    if (err) {
                        return console.error(`Error inserting default proxy_list row: ${err.message}`);
                    }
                    // console.log(`Inserted default row into proxy_list with ID: ${this.lastID}`);
                });
            } else {
                // console.log('Row with ID 1 already exists in proxy_list.');
            }
        });
    });
}

// Fungsi pushProxy untuk memasukkan proxy ke dalam database
const pushProxy = () => {
    // Pastikan file database ada
    ensureDatabaseExists();

    // Membuka koneksi ke database SQLite
    const db = new sqlite3.Database(dbFilePath);

    // Pastikan tabel-tabel sudah ada
    ensureTablesExist(db);

    ensureProxyListDefault(db);

    // Baca file proxyList.txt
    const fileStream = fs.createReadStream(path.join(__dirname, '../proxy/proxyList.txt'));

    const rl = readline.createInterface({
        input: fileStream,
        crlfDelay: Infinity
    });

    // Nilai default untuk tabel proxies
    const proxy_list_id = 1;
    const used = 0;
    const status = 'available';

    let queriesPending = 0; // Untuk melacak jumlah query yang berjalan

    // Fungsi untuk memasukkan data proxy ke dalam database
    function insertProxy(address) {
        const query = `INSERT INTO proxies (proxy_list_id, address, used, status) VALUES (?, ?, ?, ?)`;
        queriesPending++; // Tambahkan jumlah query yang berjalan
        db.run(query, [proxy_list_id, address, used, status], function (err) {
            if (err) {
                console.error(`Error inserting proxy: ${err.message}`);
            } else {
                // console.log(`Proxy inserted with ID: ${this.lastID}`);
            }
            queriesPending--; // Kurangi jumlah query yang berjalan
            if (queriesPending === 0) {
                closeDatabase(); // Tutup database jika tidak ada query yang tersisa
            }
        });
    }

    // Membaca setiap baris dari file dan masukkan ke dalam database
    rl.on('line', (line) => {
        insertProxy(line.trim());
    });

    // Fungsi untuk menutup database setelah semua query selesai
    function closeDatabase() {
        console.log('All queries finished.');
        db.close((err) => {
            if (err) {
                return console.error(`Error closing database: ${err.message}`);
            }
            console.log('Database connection closed.');
            process.exit(0);
        });
    }

    // Pastikan menutup database jika sudah selesai membaca file
    rl.on('close', () => {
        console.log('Finished reading proxyList.txt');
        if (queriesPending === 0) {
            closeDatabase(); // Jika tidak ada query yang tersisa, tutup database
        }
    });
};

// Ekspor fungsi pushProxy agar bisa dipanggil dari file lain
module.exports = {
    pushProxy
};